/*
**	HTV.p			Nov. 2004
**
**	by Peter Hrdahl, Oreste Tristani and David Vestin (European Central Bank)
**
**	--------------------------------------------------------------------------------------------------------------------------------
**
**	Gauss program implementing the joint macro-term structure model of Hrdahl, Tristani and Vestin (Journal of Econometrics, 2005).
**
**	This program uses the Sderlind (1999) RE solving routines and Goffe's Simulated Annealing routine written by E.G.Tsionas.
**	The program was written for Gauss version 6.0. It may be used freely for non-commercial, academic purposes and as long as the 
**	authors are credited. No responsibilities for the program's actual performance can be assumed.
**	
*/

new;
cls;
#include Fuhr.prc;					/* for setting up simple model*/
#include PsTgSen.prc;					/* for generalized Schur decomposition*/
#include PsGees.prc;					/* for Schur decomposition*/
#include MonEEA.prc;					/* for solution algorithms*/
#include anneal;					/* for simulated annealing method */
dlibrary -a PsdGees;					/* making dll available*/

/*********************************************************************************************************************
		 			Data & initial definitions
**********************************************************************************************************************/

load y[] = byld-7y.txt;					/* zero-coupon yields (% p.a., continuously compounded) */
tau = 1/12 | 0.25 | 0.5 | 1 | 3 | 7 ;	/* maturities of bonds (years) */
Uindex = 2 | 3 | 4 | 6;					/* position of unobservable yields */
Oindex = 1 | 5;					    	/* position of observable yields */
h = 1/12;						        /* time increment (years) */
load md[] = macr-yrt.txt;				/* macro data: inflation and output gap */
K1 = 2;			        				/* Number of macro variables */
K2 = 4;					        		/* Number of unobservable variables */
K3 = 1;							        /* Number of additional variables (lagged r) */
p = 3;				        			/* Number of lagged non-predetermined variables */
q = 11;						        	/* Number of leads of non-predetermined variables  */
n1 = K1*p + K2 + K3;
n2 = K1*(q+1);

/*********************************************************************************************************************
			 			 Initial parameters & bounds
**********************************************************************************************************************/

load param = .......;					/* Vector of initial parameter values */
nLam = 14;							    /* Number of price-of-risk parameters (lambda) to be estimated */

/* parameter bounds */
Bounds = 	0 ~ .999 |			    	/* rho 				*/
		-100 ~ 100  |				    /* beta 			*/
		-100 ~ 5 |	        			/* gamma			*/
		0 ~ 1 |					        /* mu(pi) 			*/
		ones(2,1).*(-2 ~ 2) |			/* delta(pi) 			*/
		-1 ~ 1 |			        	/* delta(x) 			*/		
		0 ~ 1  |			        	/* mu(x) 			*/
		ones(3,1).*(-2 ~ 2) |			/* zeta(x) 			*/
		-1 ~ 1 |			        	/* zeta(r) 			*/
		-0.999 ~ .999 |		    		/* fi(pi*)			*/
		ones(4,1).*(0 ~ 1) |			/* sigma (pi*,eta,AS,IS)	*/
		ones(4,1).*(0 ~ 1) |			/* sigma-m 			*/
		ones(nLam,1).*(-10000000 ~ 10000000) ;	/* lambda 			*/

/*********************************************************************************************************************
						Simulated Annealing input parameters
**********************************************************************************************************************/

sa_t = 30;						/* Temperature */
rt = .9;
neps = 6;
max = 1;
eps = 1.0E-6;
ns = 20;
nt = 100;
maxevl = 10000000000;
iprint = 1;
npar = rows(param);
sa_c = 2.0*ones(npar,1);
lb = Bounds[.,1];
ub = Bounds[.,2];
vm = 0.01*ones(npar,1);
rndseed 123;

/*********************************************************************************************************************
			 			 Yield data
**********************************************************************************************************************/

ny = rows(tau);                         /* # of yields */
y = reshape(y,rows(y)/ny,ny)/1200;
meany1 = meanc(y[2:rows(y),1]);
y=y-meany1;                             /* Subtracts the mean of the short rate from all yields */
Yp = y[2:rows(y),Oindex];               /* Perfectly observable yields */
Ym = y[2:rows(y),Uindex];               /* Yields observed with error */
m = cols(Ym);                           /* # of unobservable yields */
t = seqa(1/12,1/12,12*maxc(tau));       /* Time to maturity vector (monthly)  */
t2 = t/h;
N = rows(t);                            /* Number of maturities in t-vector */
indx = 1 | 3 | 6 | 12 | 36 | 84 ;       /* Index of positions in t-vector corresponding to tau */
CapT = rows(y)-1;                       /* # of obs. -- minus 1 because of the lagged interest rate in the policy rule  */

/*********************************************************************************************************************
			 			 Macro data
**********************************************************************************************************************/

md = reshape(md,rows(md)/2,2);
md = trimr(md,1,0);                     /* Shortens sample due to lagged policy rate */
dp = md[.,1];                           /* inflation */
meandp = meanc(dp[13:rows(dp)]);        /* average inflation in the data used for estimation */
dp = dp - meandp;                       /* de-means dp */
lcgap = md[.,2]/12;                     /* output gap ( /12 ) */
meangap = meanc(lcgap[13:rows(lcgap)]); /* average output gap in the data used for estimation */
lcgap = lcgap - meangap;                /* de-means lcgap */

/* Current inflation&gap and p lags */
Xdp = trimr(dp,12,0);
Xlcgap = trimr(lcgap,12,0);
i = 1;
do while i le p;
	Xlcgap = Xlcgap ~ trimr(lcgap,(12-i),i);        /* max of 12 lags possible */
	Xdp = Xdp ~ trimr(dp,(12-i),i);
	i = i + 1;
endo;
lag_r = trimr(y[.,1],0,1);                          /* lagged short rate */
XO = Xlcgap[.,2:p+1] ~ Xdp[.,2:p+1] ~ lag_r ;       /* Observable predetermined variables */
X2bar = Xlcgap[.,1] ~ Xdp[.,1] ;                    /* Observable non-predetermined variables */ 
YpX2 = Yp ~ X2bar;                                  /* Perfectly observable variables (t-dated) */

/*********************************************************************************************************************
						clears globals
**********************************************************************************************************************/

clear lambda0,lambda1,sigmavec,AP,Am,BO,BU,lnL,i,A,B,shrtrate,z,Sig_exog,CxMbar,CxMbarO,CxMbarU,MSd,CSd;
clear rho,beta,gam,mu_pi,dvec_pi,delta_x,mu_x,zvec_x,zeta_r,fi_pi,sig_pi,sig_eta,sig_AS,sig_IS;
clear ASd11,ASd12,ASd21,ASd22,ASd,BSd1,BSd2,BSd,FSd;
clear U_denom,BO_CO,AP_0,Xt_hat,BUm,BOm,Btilde,Sig2inv,TopA11,A22bot,inv1,s2;
clear A,B,Abar,Bbar,MSd2,Dhat,DhatInv,Deltap,z1,z2,z3,z4,z5,z6,z7,z8,z9,z10,o1,o2,Dhat1,Dhat2,XU,um;
penalty = -200000;

/*********************************************************************************************************************
						Procedures
**********************************************************************************************************************/

/*** Defines lambda matrices ***/
Proc (0) = DefLambda(par);
	lambda0 = par[z+1:z+4] ;		z = z+4;
	lambda1 =	0 ~ 0 ~ par[z+1] ~ par[z+2] |
			par[z+3] ~ par[z+4] ~ par[z+5] ~ 0 |
			par[z+6] ~ par[z+7] ~ par[z+8] ~ 0 |
			0 ~ par[z+9] ~ 0 ~ par[z+10] ;
endp;

/*** Defines parameters ***/
Proc (0) = DEFPARAM(par);
	/* Policy rule parameters */
	rho = par[1];
	beta = par[2];
	gam = par[3];
	/* Philips curve */
	mu_pi = par[4];
	dvec_pi = par[5] | par[6] | (1 - sumc(par[5:6]));
	delta_x = par[7];
	/* IS equation */
	mu_x = par[8];
	zvec_x = par[9:11];
	zeta_r = par[12];
	/* Unobservables */
	fi_pi = par[13];
	/* SdDev of shocks to unobservables */
	sig_pi = par[14];
	sig_eta = par[15];
	sig_AS = par[16];
	sig_IS = par[17];
	/* SdDev of measurement errors */
	sigmavec = par[18:17+m];
	/* Prices of risk */
	z = 17+m;
	DefLambda(par);
endp;

/*** Solves the REE using Sderlind's algorithm ***/
proc (2) = SimpRul2( A,B,n1,n2,F,Cutoff);    		/* Taken from SimpRulT in MonEEA.prc */
    local T,Z,lambda,logcon,Zkt,Zlt,Ttt,oldtrap,Zkt_1,C,M;
	A = A - B*F;
	{ T,Z,lambda } = Dgees( A,CutOff,0 );
	logcon = packr(abs(lambda)) .<= cutoff;
	if sumc(logcon) ne n1;
		M = -999;
		C = -999;
	else;
		Zkt = Z[1:n1,1:n1];
		Zlt = Z[n1+1:n1+n2,1:n1];
		Ttt = T[1:n1,1:n1];
		oldtrap = trapchk(1);
		trap 1,1;            
		Zkt_1 = inv(Zkt);
		C = Zlt*Zkt_1;              /* x2(t) = C*x1(t)*/
		M = Zkt*Ttt*Zkt_1;          /* x1(t+1) = M*x1(t) + e(t+1)*/
		M = trunc(M * 1e+8) * 1e-8;
		C = trunc(C * 1e+8) * 1e-8;
	endif;
	retp( M,C );
endp;

/*** A and B difference equations for bond pricing ***/
proc (2) = ABvectors;
	local i,Sig2,SigLam0,DhatSig;
	Abar = zeros(N,1);
	Bbar = zeros(N,n1);
	Abar[1] = 0.0;
	Bbar[1,.] = -Deltap;
	DhatSig = Dhat[7:10,7:10].*sig_exog';
	MSd2 = Dhat*Msd*DhatInv;
	MSd2[7:10,7:10] = MSd2[7:10,7:10] - DhatSig*lambda1*10000;  /* Lambdas are scaled by 10000 during estimation */
	MSd2 = trunc(MSd2 * 1e+8) * 1e-8;
	Sig2 = DhatSig*DhatSig';
	SigLam0 = DhatSig*lambda0*10000;
	i = 2;
	do while i le N;
		Abar[i] = Abar[i-1] - Bbar[i-1,7:10]*SigLam0 + 0.5*Bbar[i-1,7:10]*Sig2*Bbar[i-1,7:10]';
		Bbar[i,.] = Bbar[i-1,.] * MSd2 - Deltap;
		i = i + 1;
	endo;
	A = -Abar./t2;
	B = -Bbar./t2;
	retp(A[indx],B[indx,.]);
endp;

/*** Defines global variables ***/
proc (0) = GlobalDefinitions;
	TopA11 = zeros(1,n1) |                                  /* Top portion of A11 (called H11 in the paper) */
		( eye(p-1) ~ zeros(p-1,n1-p+1) ) |
		zeros(1,n1) |
		( zeros(p-1,p) ~ eye(p-1) ~ zeros(p-1,n1-2*p+1) ) ;
	ASd12 = ( zeros(1,q) ~ 1 ~ zeros(1,q+1) ) |	            /* A12 (called H12 in the paper) */
		zeros(p-1,n2) |
		( zeros(1,n2-1) ~ 1 ) |
		zeros(n1-p-1,n2);
	A22bot = zeros(q,q+1) ~ eye(q) ~ zeros(q,1);            /* Lower portion of A22 (called H22 in the paper) */
	BSd1 = zeros(n1-1,1) | 1;                               /* First part of K vector */
	Deltap = zeros(1,2*p+1) ~ 1 ~ zeros(1,n1-2*p-2);        /* r(t) = Deltap * Z(t)    */
	z1 = zeros(1,2*p);
	z2 = zeros(1,n1-2*p-1);
	z3 = zeros(n1-2*p-1,n1);
	z4 = zeros(1,p+3);
	z5 = zeros(q,n1);
	z6 = zeros(1,p);
	z7 = zeros(1,2);
	z8 = zeros(1,q);
	z9 = zeros(q,q+2);
	z10 = zeros(n2-1,1);
	o1 = -1*ones(1,q);
	o2 = eye(q);
	Dhat1 = eye(2*p+1) ~ zeros(2*p+1,n1-2*p-1);             /* Part of Delta-hat */
	Dhat2 = zeros(1,n1-1) ~ 1;                              /* Part of Delta-hat */

endp;

/*** Defines state-space parameter matrices ***/
proc (0) = DefSS;
	/* Define H matrix ("H" in the paper; "ASd" here) */
	ASd11 = TopA11 | (z1 ~ fi_pi ~ z2) | z3;
	ASd21 = ( -(12*(1-mu_x)/mu_x)*zvec_x' ~ z4 ~ -12/mu_x ~ 0 ) | 
		z5 |
		( z6 ~ -(12*(1-mu_pi)/mu_pi)*dvec_pi' ~ z7 ~ -12/mu_pi ~ 0 ~ 0 ) | 
		z5;
	ASd22 = ( o1 ~ 12/mu_x ~ -(12*zeta_r)/mu_x ~ z8 ) | 
		( o2 ~ z9 ) |
		( z8 ~ -(12*delta_x)/mu_pi ~ o1 ~ 12/mu_pi ) |  
		A22bot ;
	ASd = (ASd11 ~ ASd12) | (ASd21 ~ ASd22);
	/* Define K matrix ("K" in the paper; "BSd" here) */
	BSd2 = (12*zeta_r)/mu_x | z10;
	BSd = BSd1 | BSd2;
	/* Define F vector (policy rule) */
	FSd = z1 ~ beta*(1-rho) ~ -1 ~ z7 ~ -rho ~ z8 ~ -gam*(1-rho) ~ -beta*(1-rho) ~ z8;
endp;

/*** Various definitions ***/
proc (0) = Definitions;
		CxMbar = CSd[(q+1),.] | CSd[2*(q+1),.] ;                        /*  param. of the autoregr. repr. for current non-predeterm. var's  */
		shrtrate = -FSd*(eye(n1) | CSd);                                /*  param. of the autoregr. repr. for the short rate = Delta */
		Dhat =  Dhat1 | shrtrate | CSd[2*(q+1),.] | CSd[q+1,.] | Dhat2; /* Z(t) = Dhat * X1(t)    */
		DhatInv = inv(Dhat);
		Sig_exog = sig_pi | sig_eta | sig_AS | sig_IS ;                 /* StDev of unobservable state variables */
		Sig2inv = 1/(Sig_exog.*Sig_exog);
		{A,B} = ABvectors;                                              /* Defines Bond Pricing Factor Loadings */
		Btilde = B*Dhat;
		AP = A[Oindex];                                                 /* Corresponds to yields observed without measurement error */
		Am = A[Uindex];                                                 /* Corresponds to yields measured with error */
		BO = Btilde[.,1:p*K1] ~ Btilde[.,n1];                           /* Corresponds to observable states 	*/
		CxMbarO=CxMbar[.,1:p*K1] ~ CxMbar[.,n1];                        /*		"	"		*/
		BU = Btilde[.,p*K1+1:p*K1+K2];                                  /* Corresponds to Unobservable states 	*/
		CxMbarU=CxMbar[.,p*K1+1:p*K1+K2];                               /*		"	"		*/
		BUm = BU[Uindex,.];                                             /* Corresponds to yields measured with error */
		BOm = BO[Uindex,.];                                             /* Corresponds to yields measured with error */
		U_denom = (BU[Oindex,.] | CxMbarU);                             /* Jacobian matrix J */
		BO_CO = (BO[Oindex,.] | CxMbarO);
		AP_0 = AP | zeros(K1,1);
		s2 = sigmavec.*sigmavec;
endp;

/*********************************************************************************************************************
						Likelihood function
*********************************************************************************************************************/

GlobalDefinitions;

proc logL(param);
	local sum_X,sum_u;
	DEFPARAM(param);                                                    /* Defines parameters */  
	defSS;                                                              /* Defines state-space matrices */
	{MSd,CSd} = SimpRul2( ASd,BSd,n1,n2,FSd,1.0);                       /* Solves model */
	IF Msd EQ -999;
		lnL = penalty;                                                  /* Penalty for indeterminacy / explosiveness */
	ELSE;
		Definitions;
		inv1 = inv(U_denom);                                            /* inverse of the Jacobian J */
		lnL = -(CapT-1) * ( 2*ln(abs(det(U_denom))) + ln(prodc(Sig_exog.*Sig_exog)) + ln(prodc(s2)) + (K2+m)*ln(2*pi) );
		XU = inv1 * ( YpX2' - AP_0 - BO_CO*XO' );                       /* unobservables (Xu) */
		Xt_hat = XU[.,2:CapT];                                          /* Xu-errors */
		Xt_hat[1,.] = Xt_hat[1,.] - fi_pi*XU[1,1:CapT-1];               /* Xu-errors for inflation target */
		sum_X = Xt_hat.*Xt_hat;
		sum_X = sumc(sum_X');
		um = ym' - Am - BUm*XU - BOm*XO';                               /* Yield observation errors */
		sum_u = um[.,2:CapT].*um[.,2:CapT];
		sum_u = sumc(sum_u');
		lnL = lnL - Sig2inv'sum_X - (1/s2')*sum_u;
		lnL = 0.5 * lnL;                                                /* since lnL is defined as 2*log-likelihood before */
	ENDIF;
    retp(lnL);
endp;

/*********************************************************************************************************************
                                         	       Estimation
**********************************************************************************************************************/

__title = "ML estimation of HTV model";
{param, fopt, nacc, nfcnev, nobds, ier, sa_t, vm} =
      sa(&logL,param,max,rt,eps,ns,nt,neps,maxevl,lb,ub,sa_c,iprint,sa_t,vm);
      print "";
      print "****   results after sa   ****";
      print "solution" param;
      print "final step length" vm;
      print "optimal function value " fopt;
      print "number of function evaluations " nfcnev;
      print "number of accepted evaluations " nacc;
      print "number of out-of-bounds evaluations " nobds;
      print "final temperature " sa_t;
      print "error" ier;


end;