function [trend,cycle,season,v]=tcs(ndd,nc,cyc,s,x,du,rho)
% tcs ' A trend cycle season filter'  ( M. Mohr, 2005, ECB Working Paper No 499) with stochastic trend, 
%         cyclical and seasonal (Schlicht/ Pauly 1983) component.
% USAGE
% [xt,xc,{xs},{v}]=tcs(d,c,critcycl,s,x,{dum},{rho}) 
% INPUT
% d [scalar or string] level of the trend (usually in the range of 0 - 2); 
%         If d is specified as a string, then it must be of the form [d]s[seas] (without brackets) where
%                [d] is a scalar denoting the trend order and
%                [seas] is a scalar denoting the seasonality in the stochastic trend (for instance: 1s4).
%         The specification d = 0 is possible. It implies xt = 0 and makes sense only for non-trending time series.
% c [scalar] level of the 1st cyclical process (usually in the range of 0 - 2). 
%           For the HP filter or the EES, c has to be set equal to zero
% critcycl[scalar]: the critical cycle for the first stochastic AR-cylce model (> 0). 
%        If c=0 and critcycl > 0, the HP or the EES follow. In this case, the value of critcycl denotes the 
%        value of the smoothing parameter lambda.
%        c and critcycl can be both set equal to zero. Together with d > 0  this gives a 1-component trend or -if s > 0-
%        a 2-components trend-season filter.
% s[scalar]{optional}: the season: 4 for quarterly data, 12 for monthly data, etc; must be specified if xs is given
%        s > 0, together with c = 0, critcycl = 0 and d = 0 gives a 1-component seasonal filter that can be used to 
%        de-seasonalize a stationary series.
% x: row vector to be filtered 
% du{optional}: dummy matrix specifying break(s) in the trend component
% rho{optional}: scalar; dampening parameter for the stochastic AR-cycle model, optional; default is rho=0.975
% OUTPUT
% xt,xc,{xs}: row vectors with trend, cyclical, and seasonal component, respectively;
%       xs is optional.
% v{optional}: vector, contains the constant drift term if d=1 and the size of the structural breaks if du is given 
% NOTES
% See further documentation in tcs.m for examples of usage

% Denote X as the input series, T as the trend, C as the cyclical and S as the seasonal  (Schlicht/Pauly 1983) component. .
% Let D^d denote the d-th difference Matrix and L denote the lag operator.
% tcs detemines T , C and S such that objective
% (X-T-C-S)'(X-T-C-S) + T'(D^d)'D^dT+C'((I-a[1]L-a[2]L^2-...)^c)'((I-a[1]L[1]-a[2]L^2-...)^c)C+S'(I-s[1]L-s[2]L^2-...)'(I-s[1]L[1]-s[2]L^2-...)S 
% is maximised. The order of the trend process, d, and the order of the stochastic cylce, c, the average cycle length, cycle, and
%  the frequency of the  data s (to determine the season) must be specified a priori.
% One or several breaks in the trend can be accounted for by specifying a dummy matrix.
% The programme supports as special cases the HP filter and the extended exponential smoothing procedure (EES; Toedter 2003).
%
% EXAMPLES:
% [xt,xc,xs]=tcs(1,2,28,4) -> tcfilter with first order trend, 
%     second order cycle and quarterly seasonal. The average length of the cycle is 28, 
%     rho=default value of 0.975. 
% [xt,xc,xs,v]=tcs(1,2,28,4,x,dummy) -> tcfilter with season as before, but now with dummy matrix, indicating structural break(s).
% [xt,xc]=tcs(1,2,8,0,x)-> tcfilter with average cycle length of 8 periods and with rho=default of 0.975, no seasonal component
% [xt,xc,v]=tcs(1,2,8,x)-> tcfilter with average cycle length of 8 periods and with rho=default of 0.975, no seasonal component, value of drift written to v
% [xt,xc,xs,v]=tcs(1,2,8,0,x,dummy) -> tcfilter with average cycle length of 8 periods and with no seasonal component, with dummy matrix, indicating structural break(s).
% [xt,xc,v]=tcs(1,2,8,x,dummy) -> tcfilter with average cycle length of 8 periods and with no seasonal component, with dummy matrix, indicating structural break(s).
% [xt,xc,xs,v]=tcs('1s4',2,28,0,x) -> tcfilter with average cycle length of 28 periods and a quarterly unit root in the trend, no seasonal component
% [xt,xc,v]=tcs('1s4',2,28,x) -> tcfilter with average cycle length of 28 periods and a quarterly unit root in the trend, no seasonal component
% [xt,xc,xs]=tcs(2,0,0,4,x) -> tcfilter no cyclical component (since both, nc and cc =0) and a quarterly seasonal component; rho=default of 0.975
%        the cyclical component xc would be set to zero in this case
% [xt,xc,xs]=tcs(0,2,28,4,x) -> tcfilter with no trend, second order cycle and quarterly seasonal. In this way, one could filter stationary series
%        the trend component xt would be set to zero in this case
% The HP filter and the EES as special cases:
%         [xt,xc]=tcs(2,0,100,0,x) -> tcfilter with 2nd order trend and zero-order cycle, no seasonal component: this is an HP filter with lambda=100
%         [xt,xc,xs,v]=tcs(2,0,100,0,x,dummy) -> as above, but now with a dummy series or a group of dummy series, indicating structural break(s).
%         [xt,xc]=tcs(1,0,7,0,x) -> tcfilter with 1rst order trend and zero-order cycle, no seasonal component: this is an EES filter with lambda=7
%         [xt,xc,xs,v]=tcs(1,0,7,x,dummy) -> as above, but now with a dummy series or a group of dummy series, indicating structural break(s).
% A particular interesting case is the HP filter or EES, combined with a simultaneous seasonal component:
%         [xt,xc,xs]=tcs(2,0,1600,4,x) -> this is an HP filter with Lambda=1600, combined with a seasonal filter to filter out a quarterly seasonal component.
%                  In this way, the HP filter could be applied to seasonal time series. The cyclical component is computed as a residual as xc=x-xt-xs.
%                  (Applying the HP filter without a simultaneous seasonal component on a  seasonal series, i.e. [xt,xc,xs,v]=tcs(2,0,1600,0,x), 
%                  gives rise to a significant start- and end-point bias as the trend would be affected by the seasonal impact  at the beginning and 
%                  at the end of the time series). 
%       This works also for the ESS:
%       [xt,xc,xs,v]=tcs(1,0,60,4,x)
% Note:
% [xt,xc,xs,v]=tcs('1s4',2,28,4,x) -> programme break since seasonal unit root and seasonal component cannot be specified togehter.
%     The programme releases a warning in the status line
%
% NEEDS:
% dmat[from here], armatomat[from here], seasonalmat[from here], f3mat[from here]
% 
% REFERENCES:
% Mohr, M. (2005): A Trend cycle (season) filter, ECB Working paper 499, http://www.ecb.int/pub/pdf/scpwps/ecbwp499.pdf
% Schlicht, E./ R. Pauly (1983): Descriptive seasonal adjustment by minimizing perturbations, Empirica 1, p. 15-28.
% Toedter, K.-H. (2003): Exponential smoothing as an alternative to the Hodrick-Prescott filter? 
%           In I. Klein and S. Mittnick (eds.): Contributions to modern Econometrics: 
%           from data analysis to economic policy. In honor of Gerd Hansen, pp 223-237, Kluwer.
%
% DISCLAIMER:
% THIS PROGRAMME MAY BE USED AND DISTRIBUTED FREELY.
% THE USER USES THIS MATLAB PROGRAMME AT HER OR HIS OWN RISK.
% NEITHER THE AUTHOR OF THIS PROGRAMME NOR THE ECB WILL BE RESPONSIBLE FOR ANY DAMAGE
% TO DATA, SOFTWARE, HARDWARE OR ANYTHING ELSE CAUSED BY USE OF THIS MATLAB
% PROGRAMME.
%------------------------------------------------------------------------------------------------
noseason=0;
if nargin < 7 % the user has not specfied rho
    rho=0.975; % rho takes the default value
end
if size(s,1) > 1  % the user has not specified season
    noseason=1;
    season=[];
    if nargin > 4  % with dummy 
        if nargin > 5 & size(du,1)==1 %rho given by user
               rho=du; 
        end
        du=x;
    end
    x=s;
    s=0;
end
N=size(x,1);
trs=1;  % trs: seasonal difference; trs=1 -> trend does not contain seasonal root 
if isnumeric(ndd) % trend does not contain a seasonal root 
    nd=ndd;
    trs=1; 
else  % trend contains a seasonal root 
    [ndstr, trsstr]=strtok(lower(deblank(ndd)),'s');
    trsstr=strrep(trsstr,'s','');
    nd=str2num(ndstr);   % nd: order of trend
    trs=str2num(trsstr); % trend contains seasonal root with seasonal difference trs
    noseason=1;
    season=[];
end
if s > 1 & trs > 1 % programme halt if both, seasonal component and seasonal roots specified
    warning( ' ! ERROR: Seasonal roots and seasonal component at the same time not possible. Programme halted ! ');
    trend=[];
    cycle=[];
    season=[];
    v=[];
    return
end
if nd > 0 % order of trend > 0 -> stochastic trend is specified
    D=dmat(N,trs,nd);
    U=[]; % U contains dummies and/or unity vector if nd = 1
    if nd==1 % nd = 1 -> first order stochastic trend with deterministic drift 
        U=[U ones(N,1)]; % U contains unity vector to capture the deterministic drift constant
        U(1)=0;
    end
    if  exist('du') & ~isempty(du)  % structural breaks in the trend specified by dummy matrix du 
        U=[U D*du]; % write dummy matrix du into U
    end
    if exist('U') & ~isempty(U) % U exists and is not empyt -> either structural breaks or constant drift term
        W=U*inv(U'*U)*U'; 
        T=D'*(eye(N)-W)*D;
    else % neither structural breaks nor drift 
         T=D'*D;
    end
else
    D=zeros(N); % order of trend = 0 -> no trend specified
    T=0;
end
if nc <= 0 & cyc > 0 % this captures the case of the HP filter or the EES
    T=cyc*T;  % if nc<= 0 and cyc > 0, cyc contains the value of lambda of the HP filter or the EES
end
if nc > 0 % nc > 0 -> stochastic cycle is specified
    mue=2*pi/cyc; 
    arma=[1 -2*rho*cos(mue) rho^2; 1 -rho*cos(mue) 0]; % create A and B, the ar ant ma matrices of the stochastic cycle
    [A, B]=armatomat(arma,N); 
    A=A^nc; 
    B=B^nc;
    A(1:2*nc,:)=[];
    B(1:2*nc,:)=[];
    C=A'*inv(B*B')*A; % create C
else
    C=0; % order of cycle =0 -> no stochastic cycle specified 
end
if s > 1 % if a sesonsal process was specified, compute the matrix S for the stochastic seasonal process
       seasm=seasonalmat(s); 
       [P, Q]=armatomat(seasm,N);
       P(1:s-1,:)=[];
       Q(1:s-1,:)=[];
       S=P'*inv(Q*Q')*P;
else
        S=0;   % s<1 -> 2-component seasonal filters collapse to NxN zero matrix    
end
% now compute the 3-component filter matrices
MTCS=f3mat(T,C,S);
MSTC=f3mat(S,T,C);
if nc > 0 
    MCTS=f3mat(C,T,S);
else % order of stochastic cycle zero,
    if (cyc > 0) % but cycle > 0: this is the case of the HP filter or the EES
        MCTS=eye(N)-MTCS-MSTC; % which means that the cycle is just the residual
    else % and cycle <=0: there is no cycle 
        MCTS=0;
    end
end
trend=MTCS*x; % compute the trend component
cycle=MCTS*x; % compute the cyclical component
season=MSTC*x; % compute the seasonal component
v=[];
if ~isempty(U) % no-emptu U means that 
    % there is a constant drift (i.e. nd=1) or there are structural breaks
    % -> compute the drift (if there is one) and the size of  the structural breaks 
    if noseason  % no seasonal component specified
        season = inv(U'*U)*U'*D*trend;  % -> drift and structural breaks written into season
    else  % seasonal component specified
        v= inv(U'*U)*U'*D*trend;  % -> drift and structural breaks written into v
    end
else
    v=[];
end
%============================================================================

%============================================================================
function f1m=f1mat(A)
%----------------------------------------------------------------------------
% provides 1-component filter matrix MA
%----------------------------------------------------------------------------
N=size(A,2);
if (A==zeros(N))
    f1m=0;
else
    f1m=inv(eye(N)+A);
end
%============================================================================

%============================================================================
function f2m=f2mat(A,B)
%----------------------------------------------------------------------------
% provides 2-component filter matrix MAB
%----------------------------------------------------------------------------
N=size(A,2);
if (A==zeros(N))
    f2m=0;
else
    MA=f1mat(A);
    MB=f1mat(B);
    f2m=inv(eye(N)-MA*MB)*MA*(eye(N)-MB);
end
%============================================================================

%============================================================================
function f3m=f3mat(A,B,C)
%----------------------------------------------------------------------------
% provides 3-component filter matrix MABC
%----------------------------------------------------------------------------
N=size(A,2);
if (A==zeros(N))
    f3m=0;
else
    MAB=f2mat(A,B);
    MCB=f2mat(C,B);
    f3m=inv(eye(N)-MAB*MCB)*MAB*(eye(N)-MCB);
end
%============================================================================

%============================================================================
function seasm=seasonalmat(s)
%----------------------------------------------------------------------------
% Builds a 2xs matrix with the parameters of the Pauly/Schlicht (Empirica 1983) seasonal ARMA process
% The first row of this matrix contains the vector of ar-parameters and
% the second row contains the vector of ma-parameters.
% The implied ARMA process is p(L)x_s=q(L)e, where e is white noise and x_s is the seaonal process
% p(L):=sum[L^i,{i,0,s-1}], q(L):=(1/s)Sum[(s - i) L^(i - 1), {i, 1, s - 1}]
% The parameters for lag zero appear in the first column of the arma matrix.
% s[scalar]: the seasonal frequency (i.e. s for quarterly data, 12 for monthly data)
% seasmmatrix]: the ARMA process output matrix The first row contains the AR parameters, 
% the second row contains the MA params. Note that the parameters 
% for lag zero appear in the first column of ARMA
% Needs:
% nothing
%--------------------------------------------------------------------------
if s==0
   seasm=0;
else
    seasm=[ones(1,s);(s-(1:s))/s];
end
%============================================================================

%============================================================================
function [ar,ma]=armatomat(arma,T)
%----------------------------------------------------------------------------
% Builds an ar matrix and an ma matrix out of a 2xk matrix
% which first row contains the vector of ar-parameters and
% which second row contains the vector of ma-parameters.
% The implied ARMA process is A(L)y=B(L)e, where e is white noise
% Note that the arma matrix includes the parameters for lag zero in the first column
% which - in most cases - will be equal to one
% arma[matrix]: 2xk matrix. The first row contains the AR params, 
% the second row contains the MA params. Note that the parameters 
% include the parameters for lag zero in the first column of arma
% and will ususally be equal to 1
% T[scalar]: Integer, size of the ar and ma matrices
% ar[matrix]: the ar output matrix 
% ma[matrix]: the ma output matrix 
% Needs:
% dmat[from here]
%--------------------------------------------------------------------------
[n k]=size(arma);
k=k-1;
di=0:-1:-k;
ar=kron(ones(T,1),arma(1,:));
ar=spdiags(ar,di,T,T);
ar(1:k,1:k)=0;
if n > 1
    ma=kron(ones(T,1),arma(2,:));
    ma=spdiags(ma,di,T,T);
else 
    ma=eye(T);
end
%============================================================================

%============================================================================
function lm=lmat(T,n)
%--------------------------------------------------------------------------
% Description
% Gives a TxT lag matrix Lmat with lag n such that 
% lmat(T)*X gives n-lagged values of X (X being an arbitrary vector)
% Usage:
% lmat(T, n)
% T[scalar]: Integer, Dimension of the TxT output matrix
% n[scalar]: Integer, lag length
% Needs:
% Nothing
%--------------------------------------------------------------------------
lm=diag(ones(1,T-n),-n);
%============================================================================

%============================================================================
function dm=dmat(T,s,n)
%----------------------------------------------------------------------------
% Gives a TxT nth-difference matrix dmat such that 
% dmat*X produces the n-th Differences of season sss of X (X being an arbitrary vector): (1-L^[sss])^n X[t]
% T[scalar]: Integer, dimension of the TxT output matrix
% s[scalar]: Integer, order of season (e.g.: 1 for annual data, 4 for quarterly data)
% n[scalar]: Integer, order of difference
% Needs:
% lmat[from here]
%--------------------------------------------------------------------------     
L=lmat(T,s);
DD=(eye(T)-L)^n;
L=lmat(T,s*n);
dm=L*L'*DD;
%============================================================================