'============================================================================
' TCS
'-------------------------------------------------------------------------------------------------
' Description:
' tcs ' A trend cycle season filter( M. Mohr, 2005) with stochastic trend,cyclical and seasonal (Schlicht/ Pauly 1983) component. 
' Denote X as the input series, T as the trend series and C and S as the cycle and the seasonal series.
' Let D^d denote the d-th difference Matrix and L denote the lag operator
' TCS detemines T and C such that objective
' (X-T-C-S)'(X-T-C-S) + T'(D^d)'D^dT+C'((I-a[1]L-a[2]L^2-...)^c)'((I-a[1]L[1]-a[2]L^2-...)^c)C+S'(I-s[1]L-s[2]L^2-...)'(I-s[1]L[1]-s[2]L^2-...)S 
' is maximised. The order of the trend process, d, and the order of the stochastic cylce, c, 
' the average cycle length, cycle, the frequency of the data s (to determine the season) must be specified a priori.
' One break or several breaks in the trend can be accounted for by specifying a dummy series or a group of dummy series.
' The programme supports as special cases the HP filter or the extended exponential smooting procedure (EES; Toedter 2003).
'
' Usage: 
' run tcs d c critcycl {s} x xt xc {xs} {dum} {rho} 
'        (Make sure that x does not contain any NAs in the current sample !)
' d [scalar or string] level of the trend (usually in the range of 0 - 2); 
'         If d is specified as a string, then it must be of the form [d]s[seas] (without brackets) where
'                [d] is a scalar denoting the trend order and
'                [seas] is a scalar denoting the seasonality in the stochastic trend (for instance: 1s4).
'         The specification d = 0 is possible. It implies xt = 0 and makes sense only for non-trending time series.
' c [scalar] level of the 1st cyclical process (usually in the range of 0 - 2). For the HP filter or the EES, c has 
'         to be set equal to zero
' critcycl[scalar]: the critical cycle for the first stochastic AR-cylce model (> 0). 
'        If c=0 and critcycl > 0, the HP or the EES follow. In this case, the value of critcycl specifies the value of the smoothing 
'        parameter lambda.
'        c and critcycl can be both set equal to zero. Together with d > 0 this gives a 1-component trend or -if s > 0- 
'        a 2-components trend-season filter.
' s[scalar]{optional}: the season: 4 for quarterly data, 12 for monthly data, etc; must be specified if xs is given
'        s > 0, together with c = 0, critcycl = 0 and d = 0 gives a 1-component seasonal filter that can be used 
'        to de-seasonalize a stationary series.
' x[string]: name of series to be filtered 
' xt[string]: name of trend series
' xc[string]: name of the cyclical component 
' xs[string]{optional}: name of the seasonal component; must be specified if s is given
' dum[string]{optional}: name of the dummy series or group of series specifying break(s) in the series
' rho[scalar]{optional}: the dampening parameter for the stochastic AR-cylce model, optional; default is rho=0.975
'
' Examples:
' run tcs 1 2 28 4 y yt yc ys  -> tcfilter with first order trend, 
'     second order cycle and quarterly seasonal. The average length of the cycle is 28, 
'     rho=default value of 0.975. 
' run tcs 1 2 28 4 y yt yc ys  dummy -> tcfilter with season as before, but now with a dummy series or a group of dummy series, indicating structural break(s).
' run tcs 1 2 8 y yt yc -> tcfilter with average cycle length of 8 periods and with rho=default of 0.975, no seasonal component
' run tcs 1 2 8 y yt yc dummy -> tcfilter with average cycle length of 8 periods and with rho=default of 0.975, no seasonal component, with a dummy series or a group of dummy series, indicating structural break(s).
' run tcs 1 2 8 0 y yt yc -> tcfilter with average cycle length of 8 periods and with rho=default of 0.975, no seasonal component (since seas=0)
' run tcs 1 2 8 y yt yc 0.985 -> tcfilter with average cycle length of 8 periods and rho=0.985, no seasonal component
' run tcs 1s4 2 28 y yt yc -> tcfilter with average cycle length of 28 periods and with rho=0.975 and a quarterly unit root in the trend, no seasonal component
' run tcs 2 0 0 4 y yt yc ys-> tcfilter no cyclical component (since both, nc and cc =0) and a quarterly seasonal component; rho=default of 0.975
'        the cyclical component yc would be set to zero in this case
' run tcs 0 2 28 4 y yt yc ys  -> tcfilter with no trend, second order cycle and quarterly seasonal. In this way, one could filter stationary series
'        the trend component yt would be set to zero in this case
' The programme contains the HP filter and the EES as special cases:
'         run tcs 2 0 100 y yt yc -> tcfilter with 2nd order trend and zero-order cycle, no seasonal component: this is an HP filter with lambda=100
'         run tcs 2 0 100 y yt yc dummy -> as above, but now with a dummy series or a group of dummy series, indicating structural break(s).
'         run tcs 1 0 7 y yt yc -> tcfilter with 1rst order trend and zero-order cycle, no seasonal component: this is an EES filter with lambda=7
'         run tcs 1 0 7 y yt yc dummy -> as above, but now with a dummy series or a group of dummy series, indicating structural break(s).
' A particular interesting case is the HP filter or EES, combined with a simultanoues seasonal component:
'         run tcs 2 0 1600 4 y yt yc ys-> this is an HP filter with Lambda=1600, combined with a seasonal filter to filter out a quarterly seasonal component.
'                  In this way, the HP filter could be applied to seasonal time series. The cyclical component is computed as a residual as yc=y-yt-ys.
'                  (Applying the HP filter without a simultaneous seasonal component on a  seasonal series, i.e. run tcs 2 0 1600 y yt yc, 
'                  gives rise to a significant start- and end-point bias as the trend would be affected by the seasonal impact  at the beginning and 
'                  at the end of the time series). 
'       This works also for the ESS:
'       run tcs 1 0 60 4 y yt yc ys
' Note:
' tcs 1s4 2 28 4 y yt yc ys  -> programme break since seasonal unit root and seasonal component cannot be specified togehter.
'     The programme releases a warning in the status line
'
' Output:
' trend series xt[series], 
' cyclical series xc[series]
' seasonal component xs[series]
' _$v[matrix(n,1)] gives the size of the n breaks in the trend (if the user has specified any by dum)
'
' Needs:
' tmcs[from here], seasonalmat[from here]
'
' REFERENCES:
' Mohr, M. (2005): A Trend cycle (season) filter, ECB Working Paper 499, http://www.ecb.int/pub/pdf/scpwps/ecbwp499.pdf
' Schlicht, E./ R. Pauly (1983): Descriptive seasonal adjustment by minimizing perturbations, Empirica 1, p. 15-28.
' Toedter, K.-H. (2003): Exponential smoothing as an alternative to the Hodrick-Prescott filter? 
'           In I. Klein and S. Mittnick (eds.): Contributions to modern Econometrics: 
'           from data analysis to economic policy. In honor of Gerd Hansen, pp 223-237, Kluwer.
'
' DISCLAIMER:
' THE USER USES THIS MATLAB PROGRAMME AT HER OR HIS OWN RISK.
' NEITHER THE AUTHOR OF THIS PROGRAMME NOR THE ECB WILL BE RESPONSIBLE FOR ANY DAMAGE
' TO DATA, SOFTWARE, HARDWARE OR ANYTHING ELSE CAUSED BY USE OF THIS MATLAB
' PROGRAMME.
'------------------------------------------------------------------------------------------------
!rho=0.975
!pi=@acos(-1)
!trseas=1
if @val(%0)<>NA then
    !trseas=1
    !nd = @val(%0)
else
    if @mid(%0,2,1)="s" then
        !nd = @val(@left(%0,1))
        if @val(@mid(%0,3,@strlen(%0)-2))<> NA then
            !trseas=@val(@mid(%0,3,@strlen(%0)-2))
        else
            !trseas=1
        endif
    endif
endif
!nc = @val(%1)
!cc=@val(%2)
if !cc>0 then
    !mue=2*!pi/!cc
endif

if @val(%3) = NA  then 'no seasonal component
        !seas=0
        vector y_=   {%3}
        %trend=%4
        %cyc=%5
        if @val(%6) <> NA then
            !rho=@val(%6)
            if  %7 <> "" then   
                matrix du_={%7}
            else
                matrix du_= @filledmatrix(@rows(y_),1,0) 
            endif
        else 
            if %6  <> "" then
                matrix du_={%6}
            else
                matrix du_= @filledmatrix(@rows(y_),1,0) 
            endif
        endif
else
        !seas=@val(%3)
        vector y_=   {%4}
        matrix du_= @filledmatrix(@rows(y_),1,0) 
        %trend=%5
        %cyc=%6
        %sea=%7
        if @val(%8) <> NA then
            !rho=@val(%8)
            if %9 <> ""then     
                matrix du_={%9}
            else
                matrix du_= @filledmatrix(@rows(y_),1,0) 
            endif
        else 
            if %8 <> "" then
                matrix du_={%8}
            else
                matrix du_= @filledmatrix(@rows(y_),1,0) 
            endif
        endif
endif
if (!trseas >1 and !seas > 0) then
    statusline ! ERROR: Seasonal unit roots and seasonal component at the same time not possible. Programme halted !
    stop
endif
matrix(2,3) _armacyc
if !nc>0 then
       if !cc > 0 then
             _armacyc.fill(b=r) 1,-2*!rho*cos(!mue),!rho^2,1,-!rho*cos(!mue),0
       else
         !cc = 0 means that no cyclical component is given (only trend and possibly season!)
         ' the formal implications are MC=0, MTC=MT, and MSC=MS
             !nc=-1
      endif
else     
    '!nc <= 0 means that no cyclical component is given (only trend and possibly season!)
    ' the formal implications are MC=0, MTC=MT, and MSC=MS 
     if !cc > 0 then    
         '!nc <= 0 but !cc > 0 depicts the hp filter and !cc contains the value for lambda.
         'This is indicated by !nc>1000, whereby !nc=1000+lambda 
         !nc=!cc+1000
     else
         !nc=-1
    endif
endif
matrix _armaseas
call seasonalmat(!seas,_armaseas)
vector(@rows(y_)) yc_
vector(@rows(y_)) ys_
vector(@rows(y_)) yt_
matrix _$V
call tmcs(_armacyc,_armaseas,!nd,!trseas,!nc,y_,yt_, yc_,ys_,du_,_$V)
mtos(yt_, {%trend})
mtos(yc_, {%cyc})
if  !seas>0 then
    mtos(ys_, {%sea})
endif
delete _armacyc _armaseas y_ yt_ yc_ ys_ du_

'============================================================================
subroutine tmcs(matrix arma, matrix seasm, scalar nd, scalar trendseason, scalar nc, vector y, vector yt, vector yc,vector ys, matrix dum, matrix v)
'-------------------------------------------------------------------------------------------------
' Description:
' Filters a vector using the tcs filter with 1 stochastic cylce of order nc and one stochastic seasonal component. 
'
' usage: 
' call tmcs( armacyc,armaseas, nd, trseas, nc,y,yt, yc,ys)
' armacyc[matrix] 2xp matrix that contains the parameters of the ARMA cyclical process
' armaseas[matrix] 2xs matrix that contains the parameters of the ARMA seasonal process
' nd [scalar] level of the trend (usually 1 or 2); 
' trendseason[scalar]: integer scalar >0 indicating the order of the seasonal unti root in the trend (=1 for none)
' nc [scalar] order of the stocahstic cycle (usually 1 to 2)
' y[vector]: input vector to be filtered 
' yt[vector]: trend component vector 
' yc[vector]: cyclical component vector 
' ys[vector]: seasonal component vector 
' dum[vector]: dummy matrix with 0's and 1's to specify the structural break in the trend
'
' Output:
' yt[vector]: trend component vector 
' yc[vector]: cyclical component vector 
' ys[vector]: seasonal component vector 
' _$nue[vector] : size of the strucutral break(s)
'
' Needs:
'Dmat[from here], po[from here], armatomat[from here]
'------------------------------------------------------------------------------------------------
show nc
!eps=0.001
!maxiterations = 100
!disregardar = 10
!T=@rows(y)
!withma=1
if nd > 0 then
   ' create the matrix for the stochastic trend model _$D
    matrix(!T,!T)_$D
    call Dmat(!T,trendseason, nd,_$D)
    if @isobject("_$U") then
      delete _$U
    endif
    if dum <>@filledmatrix(@rows(dum),@columns(dum),0)  and nd <> 1 then 
      'define matrix _$U which accounts for structural breaks
      matrix(!T,@columns(dum)) _$U = 0
      matplace(_$U,_$D*dum,1,1)
    endif
   ' create the matrix for the stochastic trend model _$D
    if nd= 1 then 'EES! Accounting for the drift term in the random walk!
        matrix(!T,1) _$e=1
        _$e(1,1)=0
        if dum <>@filledmatrix(@rows(dum),@columns(dum),0)  then    
            'define matrix _$U which accounts for both, the drift term b and for structural breaks
             matrix(!T,@columns(dum)+1) _$U = 0
             matplace(_$U,_$D*dum,1,2)
             matplace(_$U,_$e,1,1)
        else
           'define matrix _$U which accounts for the constant drift term b (nd=1)
             matrix _$U=_$e
       endif
       delete _$e
   endif
  ' compute the outer product of the nd-Difference matrix 
   if @isobject("_$U") then
       ' if there is a drift term or there are structural breaks: 
       ' define _$W, the residual projection matrix of a regression 
       ' on the unit vector _$e and on the dummy matrix containing the structural breaks
       sym _$W=_$U*@inverse(@transpose(_$U)*_$U)*@transpose(_$U)
       ' the _$W-matrix needs then to be accounted for in the trend filter
       sym _$T=@transpose(_$D)*(@identity(!T)-_$W)*_$D 
   else
        sym _$T=@transpose(_$D)*_$D
   endif
else
    'there is no trend specified
     sym(!T)_$T 
     _$T.fill(l) 0
      matrix(!T,!T)_$D 
      _$D.fill(l) 0
endif
if !nc>1000 then
    ' the case of nc>1000 depicts the EES/HPfilter, i.e. the missing cycle specification case. In this case,   lambda=1000-!nc;
    !lambda=!nc-1000
    _$T=!lambda*_$T
endif
' create the matrix for the arma model for the cyclical process _$A
matrix(!T,!T) _$A
matrix(!T,!T) _$B
call armatomat(arma,!T, _$A, _$B)
if (nc > 0 and nc < 1000) then
     ' the case of nc>1000 depicts the EES/HPfilter, i.e. the missing cycle specification case. In this case,   lambda=1000-!nc;
   call po(_$A,nc,_$A)
   matrix _$A = @subextract(_$A,2*nc+1,1,!T,!T)
   ' compute the outer product of the matrix for the cycle
   call po(_$B,nc*!withma,_$B)
   matrix _$B = @subextract(_$B,2*nc+1,1,!T,!T)    
   sym _$C=@transpose(_$A)*@inverse(_$B*@transpose(_$B))*_$A
else 'nc <= 0 or !nc >= 1000 means that a cyclical component is not specified (only trend and possibly season!)
    sym(!T) _$C
    _$C.fill(l) 0
endif
matrix(!T,!T) _$P
matrix(!T,!T) _$Q
 !_$$s=0
call armatomat(seasm,!T, _$P, _$Q)
!_$$n=@columns(seasm)
!_$$s=0
for !_$$i = 1 to !_$$n
    if seasm(1,!_$$i) <> 0 then
        !_$$s=!_$$i
    endif
next
'compute the 2-filter matrices
if (nc > 0 and nc < 1000) then
    if nd > 0 then
        matrix _$MCT=@inverse((@identity(!T)+_$T)*_$C+_$T)*_$T
    else
        matrix _$MCT=@inverse(@identity(!T)+_$C)
    endif
else 'nc <= 0 means that a cyclical component is not given (only trend and possibly season!)
     ' the formal implications are MC=0, and MTC=MT
    if nd > 0 then
        matrix _$MCT=@inverse(@identity(!T)+_$T)
    else
        matrix(!T,!T) _$MCT
        _$MCT.fill(l) 0
    endif
endif
if !_$$s >1 then
    matrix _$P = @subextract(_$P,!_$$s,1,!T,!T)
    matrix _$Q = @subextract(_$Q,!_$$s,1,!T,!T)
    ' compute the outer product of the matrix for the season
    sym _$S=@transpose(_$P)*@inverse(_$Q*@transpose(_$Q))*_$P
    'compute some the two 2-components filter matrices for the season 
    if nd > 0 then
        matrix _$MST=@inverse((@identity(!T)+_$T)*_$S+_$T)*_$T
    else
        matrix _$MST=@inverse(@identity(!T)+_$S)
    endif
    if (nc > 0 and nc < 1000) then 
        matrix _$MSC=@inverse((@identity(!T)+_$C)*_$S+_$C)*_$C
    else  'nc <= 0 or nc>1000 mean that a cyclical component is not given (only trend and possibly season!)
        ' the formal implications are MC=0, and MSC=MS
        matrix _$MSC=@inverse(@identity(!T)+_$S)
    endif
else
    'no season: the two 2-components filter matrices for the season are zero
    sym(!T) _$S
    _$S.fill(l) 0
    matrix(!T,!T) _$MST=0
    matrix(!T,!T) _$MSC=0
endif
'compute the 3-filter matrices
if (nc > 0 and nc<1000) then
    if nd > 0 then
        matrix _$MTCS=@inverse(_$C*(@identity(!T)-_$MSC+_$T)+_$T)*_$C*(@identity(!T)-_$MSC)
    else
        matrix(!T,!T) _$MTCS  
        _$MTCS.fill(l) 0
    endif
    if nd > 0 then
        matrix _$MCTS=@inverse(_$T*(@identity(!T)-_$MST+_$C)+_$C)*_$T*(@identity(!T)-_$MST)
    else
        if !_$$s > 1 then
            matrix _$MCTS=@inverse((@identity(!T)+_$S)*_$C+_$S)*_$S
        else
            matrix _$MCTS=@inverse(@identity(!T)+_$C)
        endif
    endif
else 'nc < 0 means that neither a cyclical component nor lambda is given (only trend and season!)
     ' the formal implications are MC=0, and MCTS=0, MTCS=MTS
    matrix(!T,!T) _$MCTS=0
    if !_$$s > 1 then
        if nd > 0 then
            matrix _$MTCS=@inverse(_$S*(@identity(!T)+_$T)+_$T)*_$S
        else 
            matrix(!T,!T) _$MTCS  
            _$MTCS.fill(l) 0
        endif
   else 'nc <= 0 and !_ss=0 means that there is neitehr a cyclical component nor a seasonal component only trend)
        ' the formal implications are MTCS=MTSC=MTC=MTS=MT
        if nd > 0 then
            matrix _$MTCS=@inverse(@identity(!T)+_$T)    
        else
             matrix(!T,!T) _$MTCS  
            _$MTCS.fill(l) 0
        endif
    endif
endif
if !_$$s > 1 then
    if nd > 0 then
        matrix _$MSTC=@inverse(_$T*(@identity(!T)-_$MCT+_$S)+_$S)*_$T*(@identity(!T)-_$MCT)
    else 
        matrix _$MSTC=_$MSC
    endif
else 
    'no season: the 3-components filter matrices for the season is zero
    matrix(!T,!T) _$MSTC=0
endif
if nc>1000 then 'this is the case of the hp or ees filter. Here, cycle is simply original series minus trend
    _$MCTS=@identity(!T)-_$MTCS
    if !_$$s > 1 then
    _$MCTS=_$MCTS-_$MSTC
    endif
endif
'compute the trend
yt=_$MTCS*y
if dum <> @filledmatrix(@rows(dum),@columns(dum),0)  or nd=1 then 
    ' if there is a constant drift (i.e. if nd=1) or if there are strucutral breaks
    ' compute v which contains the drift (if there is one) and the size of the structural breaks 
    v = @inverse(@transpose(_$U)*_$U)*@transpose(_$U)*_$D*yt
    delete _$U _$W
endif
'compute the cyclical process
yc=_$MCTS*y
'compute the seasonal process
ys=_$MSTC*y
'delete stuff

delete _$A _$B _$D _$C _$S _$T _$MTCS _$MCTS _$MSTC _$MCT _$MST _$MSC _$P _$Q 

endsub
'============================================================================

'============================================================================
Subroutine po(matrix M, scalar n, matrix outm)
'--------------------------------------------------------------------------------
' Description
' Computes M.M.M. ..., n times
' Note: M^0 is defined as I, where I is the identity matrix
' with the same dimensions as M
' Usage:
' po(inmat, n, outmat)
' inmat[matrix]: square input matrix
' outmat[matrix]: output matrix
' n[scalar]:  integer, denoting the "exponent"
' Output:
' outm=M.M.M .... , n times
' Needs:
' Nothing
'--------------------------------------------------------------------------
if n=0 then
    matrix _$om=@identity(@rows(M))
else
    matrix _$om=M
endif
for !_$i = 2 to @abs(n)
    _$om=_$om*M
next
if n < 0 then 
    _$om=@inverse(_$om)
endif
outm=_$om
delete _$om
endsub
'============================================================================

'============================================================================
Subroutine Lagm(scalar T,scalar n, matrix Lmat)
'--------------------------------------------------------------------------------
' Description
' Gives a TxT lag matrix Lmat with lag n such that 
' Lmat.X gives n-lagged values of X (X being an arbitrary vector)
' Usage:
' Lagm(T, n, Lmat)
' T[scalar]: Integer, Dimension of the TxT output matrix
' n[scalar]: Integer, laglength
' Lmat[matrix]: Name of the outputmatrix 
' Output:
' LM[matrix]: Lagmatrix begin computed
' Needs:
' Nothing
'--------------------------------------------------------------------------
vector(T-n) _$vvv=1
matrix Lmat=@makediagonal(_$vvv,-n)
delete _$vvv
endsub
'============================================================================

'============================================================================
Subroutine Dmat(scalar T,scalar sss, scalar n, matrix Dm)
'----------------------------------------------------------------------------
' Description
' Gives a TxT nth-difference matrix Dmat such that 
' Dmat.X produces the n-th Differences of season sss of X (X being an arbitrary vector): (1-L^[sss])^n X[t]
' Dm(T, n, DMat)
' T[scalar]: Integer, dimension of the TxT output matrix
' sss[scalar]: Integer, order of season (e.g.: 1 for annual data, 4 for quarterly data)
' n[scalar]: Integer, order of difference
' Dmat[matrix]: Name of the outputmatrix 
' Output:
' LM[matrix]: Lagmatrix begin computed
' Needs:
' Lagm[from here], po[from here]
'--------------------------------------------------------------------------
matrix _$L
call Lagm(T,sss, _$L)
Dm=@identity(T)-_$L
call po(Dm,n,Dm)
call Lagm(T,sss*n, _$L)
Dm=_$L*@transpose(_$L)*Dm
delete _$L
endsub
'============================================================================

'============================================================================
Subroutine armatomat(matrix arma, scalar T, matrix ARMat, matrix MAMAt)
'----------------------------------------------------------------------------
' Description
' Builds an AR matrix and an MA matrix out of a 2xp Matrix
' which first row contains the vector of ar-parameters and
' which second row contains the vector of ma-parameters.
' The implied ARMA process is A(L)y=B(L)e, where e is white noise
' Note that the arma matrix includes the parameters for lag zero in the first column
' which - in most cases - will be equal to one
' Usage:
' armatomat(arma,T,infinitema, ARmat, MAmat)
' ARMA[matrix]: 2Xp matrix. The first row contains the AR params.
' The second row contains the MA params. Note that the parameters 
' include the parameters for lag zero in the first column of ARMA
' These will ususally be equal to 1
' T[scalar]: Integer, size of the AR and MA matrices
' ARmat[matrix]: the AR output matrix 
' MAmat[matrix]: the MA output matrix 
' Output:
' AR[matrix]: the AR output matrix 
' MA[matrix]: the MA output matrix 
' Needs:
' Dmat[from here], po[from here]
'--------------------------------------------------------------------------
!nrows=@rows(arma)
!ncols=@columns(arma)
vector(!nrows) _$na = 0
for !_i=1 to !nrows
    for !_j=1 to !ncols
        if arma(!_i,!_j) <> 0 then
            _$na(!_i)=!_j
        endif
    next
next
matrix ARMat=@identity(T)
matrix MAMAt=@identity(T)
if !nrows>2 then 
    !nmodels=2
else
    !nmodels=!nrows
endif
for !_m = 1 to !nmodels
    if _$na(!_m)>0 then
        matrix(T,T) _$L
        matrix(T,T) _$Z=0
        vector _$arvec
        _$arvec=@subextract(arma,!_m,1,!_m,_$na(!_m))
        for !_i=0 to _$na(!_m)-1
            call Lagm(T,!_i,_$L)
            _$Z=_$Z+_$L*_$arvec(!_i+1)
        next
        call Lagm(T,_$na(!_m)-1,_$L)
        if !_m = 1 then 
            matrix ARMat=_$L*@transpose(_$L)*_$Z
       else
            matrix MAMat=_$Z
        endif
    endif
    if @isobject("_$arvec") then 
        delete _$arvec
    endif
next
'delete stuff
if @isobject("_$L") then
    delete _$L 
endif
if @isobject("_$Z") then
    delete _$Z
endif
delete  _$na 
endsub
'============================================================================

'============================================================================
subroutine seasonalmat(scalar s, matrix outm)
'----------------------------------------------------------------------------
' Description
' Builds a 2Xs matrix with the parameters of the Pauly/Schlicht (Empirica 1983) seasonal ARMA process
' The first row of this matrix contains the vector of ar-parameters and
' the second row contains the vector of ma-parameters.
' The implied ARMA process is p(L)x_s=q(L)e, where e is white noise and x_s is the seaonal process
' p(L):=sum[L^i,{i,0,s-1}], q(L):=(1/s)Sum[(s - i) L^(i - 1), {i, 1, s - 1}]
' The parameters for lag zero appear in the first column of the arma matrix.
' Usage:
' seasonalmat(s,outm)
' s[scalar]: the seasonal frequency (i.e. s for quarterly data, 12 for monthly data)
' outm[matrix]: the ARMA process output matrix The first row contains the AR parameters, 
' the second row contains the MA params. Note that the parameters 
' Parameters for lag zero appear in the first column of ARMA
' Output:
' outm[matrix]: the ARMA process output matrix 
' Needs:
' nothing
'--------------------------------------------------------------------------
if s=0 then
    outm=0
else
    matrix(2,s) outm
    outm.fill(l) 0
    for !_$ii =1 to s
        outm(1,!_$ii)=1
    next
    for !_$ii =1 to s-1
        outm(2,!_$ii)=(s-!_$ii)/s
    next
endif
endsub
'============================================================================ 